#!/usr/bin/perl

# subroutines.pl - a set of utility functions for main-menu.pl
# Eric Lease Morgan <emorgan@nd.edu>

# 2006-01-11 - wrote pod
# 2004-10-11 - first cut

=head1 NAME

subroutines.pl - a library of functions used in a terminal-based system for doing I/O against a MyLibrary database

=head1 DESCRIPTION

The library contains ten (10) routines, each described below:

=head2 listOneFacet

This subroutine takes an integer as input, uses that integer as the id to create a new MyLibrary::Facet object, and then prints to STDOUT the object's facet_id, facet_name, and facet_note. If the input is invalid, then the output will be empty.

=cut

sub listOneFacet {

	my $id = shift;
	my $facet = MyLibrary::Facet->new(id => $id);
	print "\nFacet created/edited:\n";
	print "    id = " . $facet->facet_id . "\n";
	print "  name = " . $facet->facet_name . "\n";
	print "  note = " . $facet->facet_note . "\n";
	print "\n";

}


=head2 listAllFacets

This subroutine takes no input. It simply gets all the facets in the system and dumps each facet's facet_id, facet_name, and facet_note.

=cut

sub listAllFacets {

	# get all facets
	my @facets = MyLibrary::Facet->get_facets;
	
	# display them
	print "\nThis is a list of all the facets:\n";
	print "\tID\tName (Note)\n";
	print "\t--\t-----------\n";
	foreach my $f (@facets) { print "\t" . $f->facet_id . "\t" . $f->facet_name . " (" . $f->facet_note . ")\n" }
	print "\n";

}


=head2 listOneTerm

This subroutine takes an integer as input. It works just like listOneFacet except for term objects. It returns a human-readable list including term_id, term_name, term_note, as well as the term's parent facet_name.

=cut

sub listOneTerm {

	# display one term
	my $id    = shift;
	my $term  = MyLibrary::Term->new(id => $id);
	my $facet = MyLibrary::Facet->new(id => ($term->facet_id));
	print "\nTerm:\n";
	print "    id = " . $term->term_id     . "\n";
	print "  name = " . $term->term_name   . "\n";
	print "  note = " . $term->term_note   . "\n";
	print " facet = " . $facet->facet_name .  " (". $term->facet_id . ")\n";
	print "\n";

}


=head2 listAllTerms

This subroutine works just like listAllFacets except for term objects.

=cut

sub listAllTerms {

	# get all terms
	my @terms = MyLibrary::Term->get_terms;
	
	# display them
	print "\nThis is a list of all the terms:\n";
	print "\tID\tName (Note)\n";
	print "\t--\t-----------\n";
	foreach my $t (@terms) { print "\t" . $t->term_id . "\t" . $t->term_name . " (" . $t->term_note . ")\n" }
	print "\n";

}


=head2 listAllLocations

This routine lists all of the location types in your presently configured MyLibrary system.

=cut

sub listAllLocations {

	# get all locations
	my @locations = MyLibrary::Resource::Location->location_type(action => 'get_all');
	
	# display them
	print "\This is a list of all the locations:\n";
	print "\tID\tName (Note)\n";
	print "\t--\t-----------\n";
	foreach my $l (@locations) { print "\t" . $l . "\t" . MyLibrary::Resource::Location->location_type(action => 'get_name', id => $l) . " (" . MyLibrary::Resource::Location->location_type(action => 'get_desc', id => $l) . ")\n" }
	print "\n";

}


=head2 listOneLocationType

(I think this subroutine is broken! It is supposed to create a location object based on the input, and then output the location_id, location_name, and location_note, but as written it will create a facet object. Wrong!!)

=cut

sub listOneLocationType {

	my $id = shift;
	my $new_location = MyLibrary::Facet->new(id => $id);
	print "\Location type created/edited:\n";
	print "    id = " . $new_location->location_id   . "\n";
	print "  name = " . $new_location->location_name . "\n";
	print "  note = " . $new_location->location_note . "\n";
	print "\n";

}


=head2 listAllResources

This routine takes no input. It creates an array of all resource ids sorted by resource_name and then outputs each associated resource_id, resource_name, and resource_note.

=cut

sub listAllResources {

	# get all resources
	my @resources = MyLibrary::Resource->get_resources (sort => 'name');
	
	# display them
	print "\This is a list of all the resources:\n";
	print "\tID\tName (Note)\n";
	print "\t--\t-----------\n";
	foreach my $r (@resources) { print "\t" . $r->id . "\t" . $r->name . " (" . $r->note . ")\n" }
	print "\n";

}


=head2 listOneResource

Given a valid integer value associated with a resource object, this routine will return the resource_id, resource_name, resource_note, all of the resource's associated terms, as well as the resource's locations. Think of this routine as a dump describing everthing about a resource.

=cut

sub listOneResource {

	my $id = shift;
	my $resource = MyLibrary::Resource->new(id => $id);
	print "\nResource:\n";
	print "           id = " . $resource->id . "\n";
	print "         name = " . $resource->name . "\n";
	print "         note = " . $resource->note . "\n";
	my @resource_terms = $resource->related_terms();
	print "      term(s) = ";
	foreach (@resource_terms) {
	
		my $term = MyLibrary::Term->new(id => $_);
		print $term->term_name, " ($_)", '; ';
		
	}
	print "\n";
	my @locations = $resource->resource_locations();
	print "  location(s) = ";
	foreach (@locations) { print $_->location, "; " }
	print "\n\n";

}


=head2 escape_entities

Given a scalar as input, this function converts any special XML characters (&, <, >, ", and ') in the scalar to their entity equivelants (&amp;, &lt;, &gt;, &quot;, and &apos;). It then return the transformed scalar.

=cut

sub escape_entities {

	# get the input
	my $s = shift;
	
	# escape
	$s =~ s/&/&amp;/g;
	$s =~ s/</&lt;/g;
	$s =~ s/>/&gt;/g;
	$s =~ s/"/&quot;/g;
	$s =~ s/'/&apos;/g;

	# done
	return $s;
	
}


=head2 clearScreen

This function simply clears the terminal screen. Unfortunately, it does this in a Unix-specfic way, through the use of the clear command. This will probably break Windows.

=cut

sub clearScreen {

	# dumb! (use non-operating specific function instead)
	system ("clear");

}

=head1 SEE ALSO

manage-terms.pl, manage-resources.pl, manage-locations.pl, manage-facets.pl, main-menu.pl, index-resources.pl, images2mylibrary.pl, doaj2mylibrary.pl, search.pl, resources2swish.pl, resources2rdf.pl

=head1 AUTHOR

Eric Lease Morgan

=cut

1;

