#!/usr/bin/perl

# resources.cgi - create and maintain MyLibrary resources

# Eric Lease Morgan <emorgan@nd.edu>
# January 27, 2006 - moving to the MyLibrary distro


=head1 NAME

resources.cgi - create and manage information resources with MyLibrary


=head1 DESCRIPTION

This is the heart of this MyLibrary Portal sample application. It is used to create and manage the information resources in your system. It brings together the terms, location types, and resources you want to share.

This program also provides a means to index your MyLibrary content, and in conjunction with a supplied SRU server application, provides access to the index for searching purposes. You can also use this interface to create an OAI data repository of your content.

In this implementation, you can describe information resources using Dublin Core elements. If you need to repeat elements (like multiple subjects or creators), then consider delimiting your input with characters like verticle bars (||). For example, a subject value might be: astronomy||mathematics. When you pull out the subject values from the MyLibrary resource objects you will then be able to parse the values accordingly. Remember also that any number of facet/term combinations can be applied to any information resource.

This particular implementation does not allow information resources to have more than one location. This was done to simplify data-entry. Similarly, this particular implementation does not allow you to assign all the various characteristics to resources available from MyLibrary, but the Dublin Core elements should be enough for most people.

Finally, this is a sample application only. Libraries are expected to write their own applications for their own specific purposes.


=head1 CONFIGURATION

In order for this indexing and OAI output to work you must do a bit of configuration. In the head of the script change the values of the following three constants:


=over 4

=item 1) REBUILD

This should be defined as the full path to bin/portal2swish.sh. Unfortunately, as of this writing, swish-e does not have a Perl API for indexing content; you need to use the supplied binary. bin/portal2swish.sh is a shell script calling the binary along with a configuration file. By defining this constant you will be able to create a swish-e index from within this script. The most likely reasons why the indexing process might break are configurations or file permissions. Be watchful.


=item 2) HTDOCS

This is the full path to the root of your HTDOCS file system.


=item 3) OAIROOT

Relative to HTDOCS, this is the root to the location of the data directory where your OAI files will be written. Like the indexing process, the most likely reason your OAI output may not work as expected is because of configurations or file permissions.

If you use this script and its OAI output functionality in a production environment, then you will want to tweak the code in the subroutine called oai. Otherwise you will be outputting invalid data for your site.


=back

=head1 AUTHOR

Eric Lease Morgan


=cut


# configure
use constant REBUILD => '/usr/local/apache/htdocs/morgan/zagreb/bin/portal2swish.sh';
use constant HTDOCS  => '/usr/local/apache/htdocs/';
use constant OAIROOT => '/morgan/zagreb/oai/data/';

# require the necessary modules
use lib '../lib';
use CGI qw/:standard/;
use CGI::Carp qw(fatalsToBrowser);
use strict;
use MyLibrary::Core;
use MyLibrary::Portal;
require 'etc/subroutines.pl';

# initialize two global variables
my $cgi = CGI->new;
my $html;

# branch according to the input
if (! $cgi->param('cmd')) { &home }
elsif ($cgi->param('cmd') eq 'create')  { &create }
elsif ($cgi->param('cmd') eq 'find')    { &find }
elsif ($cgi->param('cmd') eq 'edit')    { &edit }
elsif ($cgi->param('cmd') eq 'delete')  { &delete }
elsif ($cgi->param('cmd') eq 'reindex') { &reindex }
elsif ($cgi->param('cmd') eq 'oai')     { &oai }
else { $html = $cgi->p('Unknown command. Call Eric.') }

# quit
&gracefulExit;



sub header {
	
	# get the value of this script
	my $script = $cgi->script_name;

	# return a standard html page; notice the stylesheet
	return <<EOH;
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
	<head>
		<title>Resources</title>
		<link rel='stylesheet' href='../etc/portal-screen.css' type='text/css' />
	</head>
	<body>
		<div id="navcontainer">
			<ul id="navlist">
				<li><a href="admin.cgi">Home</a></li>
				<li><a href="facets.cgi">Facets</a></li>
				<li><a href="terms.cgi">Terms</a></li>
				<li><a href="location-types.cgi">Location types</a></li>
				<li><a href="resources.cgi" id="current">Resources</a></li>
				<li><a href="patrons.cgi">Patrons</a></li>
			</ul>
		</div>
	
	<div class='menu'>
		<p><b><a href="$script">Manage resources</a></b></p>
		<a href="$script?cmd=create">Create</a><br />
		<a href="$script?cmd=find">Find</a><br />
		<a href="$script?cmd=edit">Edit</a><br />
		<a href="$script?cmd=delete">Delete</a><br />
		<br />
		<a href="$script?cmd=reindex">Make index</a><br />
		<a href="$script?cmd=oai">Make OAI</a><br />
		<br />
	</div>
	<div class='content'>
EOH

}


sub home {

	# build a simple home page
	$html  = $cgi->h1('Manage resources');
	$html .= $cgi->p('Use these pages to manage MyLibrary Portal resources.');

}


sub create {

	# initialize the output
	$html = $cgi->h1('Create');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# create a hash of location types; needs to be sorted
		my %location_type_labels;
		foreach (MyLibrary::Resource::Location::Type->all_types) {
		
			my $location_type = MyLibrary::Resource::Location::Type->new(id => $_);
			$location_type_labels{$_} = $location_type->name;
		
		}


		# build a list of terms
		my @term_names;
		foreach (MyLibrary::Term->get_terms(sort => 'name')) { push @term_names, $_->term_name }
				
		# build the form
		$html .= $cgi->p('Use this form to create a resource.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= $cgi->table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',             'Value']),
							td(['Creator/author',   $cgi->textfield(-name => 'creator')]),
							td(['Title/name',       $cgi->textfield(-name => 'name')]),
							td(['Description/note', $cgi->textarea(-name  => 'note', -rows => 7, -columns => 40)]),
							td(['Location type',    $cgi->scrolling_list(-name => 'location_type', -values => [sort values %location_type_labels])]),
							td(['Identifier/URL',   $cgi->textfield(-name => 'location')]),
							td(['Subject',          $cgi->textfield(-name => 'subject')]),
							td(['Source',           $cgi->textfield(-name => 'source')]),
							td(['Rights',           $cgi->textfield(-name => 'rights')]),
							td(['Relation',         $cgi->textfield(-name => 'relation')]),
							td(['Publisher',        $cgi->textfield(-name => 'publisher')]),
							td(['Language',         $cgi->textfield(-name => 'language')]),
							td(['Type',             $cgi->textfield(-name => 'type')]),
							td(['Format',           $cgi->textfield(-name => 'format')]),
							td(['Date',             $cgi->textfield(-name => 'date')]),
							td(['Coverage',         $cgi->textfield(-name => 'coverage')]),
							td(['Contributor',      $cgi->textfield(-name => 'contributor')]),
							td(['Facet/terms',      $cgi->scrolling_list(-name => 'term_names', -values => [@term_names], -multiple => 'true')]),
							td(['',                 $cgi->submit(-name    => 'button', -value => 'Next')])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
	elsif ($cgi->param('button') eq 'Next') {
	
		# create an object, fill it, and save
		my $resource = MyLibrary::Resource->new;
		$resource->name($cgi->param('name'));
		$resource->note($cgi->param('note'));
		$resource->subject($cgi->param('subject'));
		$resource->source($cgi->param('source'));
		$resource->rights($cgi->param('rights'));
		$resource->relation($cgi->param('relation'));
		$resource->publisher($cgi->param('publisher'));
		$resource->language($cgi->param('language'));
		$resource->type($cgi->param('type'));
		$resource->format($cgi->param('format'));
		$resource->date($cgi->param('date'));
		$resource->coverage($cgi->param('coverage'));
		$resource->contributor($cgi->param('contributor'));
		$resource->creator($cgi->param('creator'));
    	my @term_ids = ();
    	foreach ($cgi->param('term_names')) {
    	
    		my $term = MyLibrary::Term->new(name => $_);
    		push @term_ids, $term->term_id;
    		
    	}
		$resource->related_terms(new => [@term_ids]);
		my $location_type = MyLibrary::Resource::Location::Type->new(name => $cgi->param('location_type'));
		$resource->add_location(location => $cgi->param('location'), location_type => $location_type->location_type_id);
		$resource->commit;
		
		# get the term ids
		my @terms = $cgi->param('term_names');
		my @values;
		my %labels;
		foreach (@terms) {
		
			my $term = MyLibrary::Term->new(name => $_);
			push @values, $term->term_id;
			$labels{$term->term_id} = $term->term_name;
		
		}
		
		# build the form
		$html .= $cgi->p('Select zero or more terms to denote this resource as "recommended."');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= $cgi->hidden(-name => 'id', -value => $resource->id);
		$html .= $cgi->table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',        'Value']),
							td(['Facet/terms', $cgi->scrolling_list(-name => 'term_ids', -values => \@values, -default => '', -size => '', -multiple => 'true', -labels => \%labels)]),
							td(['',            $cgi->submit(-name => 'button', -value => 'Done')])
						]
					)
				 );
		$html .= $cgi->endform;
	
	}
	
	else {
    		
		# create another object
		my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
		
		# process each recommended term
		foreach ($cgi->param('term_ids')) {
		
			my $term = MyLibrary::Term->new(id => $_);
			$term->suggested_resources(new => [$resource->id]);
			$term->commit;
		
		}
		
		# get the location
		my $location;
		my @locations = $resource->resource_locations();
		foreach (@locations) { $location .= $_->location }
		
		# build a list of related terms
		my @terms = $resource->related_terms;
		my $related_terms;
		foreach (@terms) {
		
			my $term = MyLibrary::Term->new(id => $_);
			my @suggestions = $term->suggested_resources;
			my $flag = 0;
			foreach (@suggestions) {
			
				# check for recommended
				if ($_ eq $resource->id) {
				
					$flag = 1;
					last;
					
				}
				
			}
			
			if ($flag) { $related_terms .= $term->term_name . '*; ' }
			else { $related_terms .= $term->term_name . '; ' }
			
		}
		
		# echo the newly created resource
		$html .= $cgi->p('Resource created:');
		my $list = $cgi->li('ID - ' . $resource->id);
		if ($resource->creator)     { $list .= $cgi->li('Creator - '     . $resource->creator) }
		if ($resource->name)        { $list .= $cgi->li('Name - '        . $resource->name) }
		if ($resource->note)        { $list .= $cgi->li('Description - ' . $resource->note) }
		if ($location)              { $list .= $cgi->li("Location - $location") }
		if ($resource->subject)     { $list .= $cgi->li('Subject - '     . $resource->subject) }
		if ($resource->source)      { $list .= $cgi->li('Source - '      . $resource->source) }
		if ($resource->rights)      { $list .= $cgi->li('Rights - '      . $resource->rights) }
		if ($resource->relation)    { $list .= $cgi->li('Relation - '    . $resource->relation) }
		if ($resource->publisher)   { $list .= $cgi->li('Publisher - '   . $resource->publisher) }
		if ($resource->language)    { $list .= $cgi->li('Langauge - '    . $resource->language) }
		if ($resource->type)        { $list .= $cgi->li('Type - '        . $resource->type) }
		if ($resource->format)      { $list .= $cgi->li('Format - '      . $resource->format) }
		if ($resource->date)        { $list .= $cgi->li('Date - '        . $resource->date) }
		if ($resource->coverage)    { $list .= $cgi->li('Coverage - '    . $resource->coverage) }
		if ($resource->contributor) { $list .= $cgi->li('Contributor - ' . $resource->contributor) }
		$list .= $cgi->li("Terms - $related_terms");
		$list  = $cgi->ul($list);
		$html .= $list;
				
	}
    	    
}

sub find {

	# initialize the output
	$html = $cgi->h1('Find');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# build the form
		$html .= $cgi->p('Select a field and then enter a word or phrase. Enter nothing to find everything:');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'find');
		$html .= $cgi->popup_menu(-name => 'field', -values => ['name', 'description', 'term']);
		$html .= ' ';
		$html .= $cgi->textfield(-name => 'query');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'Go');
		$html .= $cgi->endform;
    
    }
    
    else {
    		
   		# initilize
   		my @resources = ();
   		my $list = '';
   		
   		# process name and description queries
   		if ($cgi->param('field') eq 'name' || $cgi->param('field') eq 'description') {
		
			@resources = MyLibrary::Resource->get_resources(field => $cgi->param('field'), value => $cgi->param('query'), sort => 'name');
			foreach (@resources) { $list .= '<li>' . $cgi->a({-href => $cgi->script_name . '?cmd=edit&button=Go&id=' . $_->id}, $_->name) . ' (' . $_->id . ')</li>' }
		
		}

		# process term queries
		else {
		
			my $term = MyLibrary::Term->new(name => $cgi->param('query'));
			if ($term) {
			
				@resources = $term->related_resources (sort => 'name');
				foreach (@resources) {
				
					my $resource = MyLibrary::Resource->new(id => $_);
					$list .= '<li>' . $cgi->a({-href => $cgi->script_name . '?cmd=edit&button=Go&id=' . $resource->id}, $resource->name) . ' (' . $resource->id . ')</li>'
					
				}
				
			}
		
		}

		# done
		$html .= $cgi->p('Your search (' , $cgi->param('query') , ') found ' , scalar(@resources) , 'record(s).');
		$html .= $cgi->ol($list);

	}
	
}


sub edit {

	# initialize
	$html = $cgi->h1('Edit resource');
	
	# branch according to the input
	if (! $cgi->param('button')) {
		
		# display a form
		$html .= $cgi->p("Enter the ID of the resource you want to edit. Don't know the ID number? Use <a href='" . $cgi->script_name . "?cmd=find'>find</a>.");
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->textfield(-name => 'id');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'Go');		 
		$html .= $cgi->endform;

	}
	
	elsif ($cgi->param('button') eq 'Go') {
	
    	# create an object based on the input, and create form parts
    	my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
    	
		# get the location
		my $location;
		my @locations = $resource->resource_locations();
		foreach (@locations) { $location .= $_->location }

		# build a list of terms
		my @term_names;
		foreach (MyLibrary::Term->get_terms(sort => 'name')) { push @term_names, $_->term_name }
				
		# build a list of terms for this resource
		my @resource_terms;
		foreach ($resource->related_terms) {
		
			my $term = MyLibrary::Term->new(id => $_);
			push @resource_terms, $term->term_name;
		
		}

		# display an edit form
		$html .= $cgi->p('Use this form to edit the resource.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->hidden(-name => 'id', -value => $cgi->param('id'));
		$html .= $cgi->table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',             'Value']),
							td(['ID',               $cgi->param('id')]),
							td(['Creator/author',   $cgi->textfield(-name => 'creator', -value => $resource->creator)]),
							td(['Title/name',       $cgi->textfield(-name => 'name', -value => $resource->name)]),
							td(['Description/note', $cgi->textarea(-name  => 'note', -value => $resource->note, -rows => 7, -columns => 40)]),
							td(['Location',         $cgi->textfield(-name => 'location', -value => $location)]),
							td(['Subject',          $cgi->textfield(-name => 'subject', -value => $resource->subject)]),
							td(['Source',           $cgi->textfield(-name => 'source', -value => $resource->source)]),
							td(['Rights',           $cgi->textfield(-name => 'rights', -value => $resource->rights)]),
							td(['Relation',         $cgi->textfield(-name => 'relation', -value => $resource->relation)]),
							td(['Publisher',        $cgi->textfield(-name => 'publisher', -value => $resource->publisher)]),
							td(['Language',         $cgi->textfield(-name => 'language', -value => $resource->language)]),
							td(['Type',             $cgi->textfield(-name => 'type', -value => $resource->type)]),
							td(['Format',           $cgi->textfield(-name => 'format', -value => $resource->format)]),
							td(['Date',             $cgi->textfield(-name => 'date', -value => $resource->date)]),
							td(['Coverage',         $cgi->textfield(-name => 'coverage', -value => $resource->coverage)]),
							td(['Contributor',      $cgi->textfield(-name => 'contributor', -value => $resource->contributor)]),
							td(['Terms',            $cgi->scrolling_list(-name => 'term_names', -values => [@term_names], -defaults => [@resource_terms], -multiple => 'true', -size => 20)]),
							td(['',                 $cgi->submit(-name    => 'button', -value => 'Next')])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
	elsif ($cgi->param('button') eq 'Next') {
	
		# create an object
		my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
		
		# get the location
		my $location;
		my @locations = $resource->resource_locations();
		foreach (@locations) { $location .= $_->location }

		# fill the resource with values and save
		$resource->name($cgi->param('name'));
		$resource->note($cgi->param('note'));
		$resource->subject($cgi->param('subject'));
		$resource->source($cgi->param('source'));
		$resource->rights($cgi->param('rights'));
		$resource->relation($cgi->param('relation'));
		$resource->publisher($cgi->param('publisher'));
		$resource->language($cgi->param('language'));
		$resource->type($cgi->param('type'));
		$resource->format($cgi->param('format'));
		$resource->date($cgi->param('date'));
		$resource->coverage($cgi->param('coverage'));
		$resource->contributor($cgi->param('contributor'));
		$resource->creator($cgi->param('creator'));
    	my @term_ids = ();
    	foreach ($cgi->param('term_names')) {
    	
    		my $term = MyLibrary::Term->new(name => $_);
    		push @term_ids, $term->term_id;
    		
    	}
		$resource->related_terms(del => [$resource->related_terms]);
		$resource->related_terms(new => [@term_ids]);
		$resource->modify_location($resource->get_location(resource_location => $location), resource_location => $cgi->param('location'));
		$resource->commit;
		
		# get the term ids
		my @values;
		my %labels;
		foreach ($cgi->param('term_names')) {
		
			my $term = MyLibrary::Term->new(name => $_);
			push @values, $term->term_id;
			$labels{$term->term_id} = $term->term_name;
		
		}
		
		# get facet/terms
		my @ids = $resource->related_terms;
		my @terms;
		foreach (@ids) {
		
			my $term = MyLibrary::Term->new(id => $_);
			my @suggestions = $term->suggested_resources;
			my $flag = 0;
			foreach (@suggestions) {
			
				# check for recommended
				if ($_ eq $resource->id) {
				
					$flag = 1;
					$term->suggested_resources(del => [$resource->id]);
					$term->commit;
					last;
					
				}
			
			}
		
			if ($flag) { push @terms, $term->term_id }
		
		}

		# build the form
		$html .= $cgi->p('Select zero or more facet/term combinations to denote this resource as "recommended."');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= $cgi->hidden(-name => 'id', -value => $resource->id);
		$html .= $cgi->table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',        'Value']),
							td(['Facet/terms', $cgi->scrolling_list(-name => 'term_ids', -values => \@values, -default => [@terms], -size => '', -multiple => 'true', -labels => \%labels)]),
							td(['',            $cgi->submit(-name    => 'button', -value => 'Done')])
						]
					)
				 );
		$html .= $cgi->endform;
	
	}
	
	else {
		
		# create another object
		my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
				
		# process each recommended term
		foreach ($cgi->param('term_ids')) {
		
			my $term = MyLibrary::Term->new(id => $_);
			$term->suggested_resources(new => [$resource->id]);
			$term->commit;
		
		}
		
		# get the location
		my $location;
		my @locations = $resource->resource_locations();
		foreach (@locations) { $location .= $_->location }
		
		# build a list of related terms
		my @terms = $resource->related_terms;
		my $related_terms;
		foreach (@terms) {
		
			my $term = MyLibrary::Term->new(id => $_);
			my @suggestions = $term->suggested_resources;
			my $flag = 0;
			foreach (@suggestions) {
			
				# check for recommended
				if ($_ eq $resource->id) {
				
					$flag = 1;
					last;
					
				}
				
			}
			
			if ($flag) { $related_terms .= $term->term_name . '*; ' }
			else { $related_terms .= $term->term_name . '; ' }
			
		}
		
		# echo the newly created resource
		$html .= $cgi->p('Resource created:');
		my $list = $cgi->li('ID - ' . $resource->id);
		if ($resource->name)        { $list .= $cgi->li('Name - '        . $resource->name) }
		if ($resource->note)        { $list .= $cgi->li('Description - ' . $resource->note) }
		if ($resource->subject)     { $list .= $cgi->li('Subject - '     . $resource->subject) }
		if ($resource->source)      { $list .= $cgi->li('Source - '      . $resource->source) }
		if ($resource->rights)      { $list .= $cgi->li('Rights - '      . $resource->rights) }
		if ($resource->relation)    { $list .= $cgi->li('Relation - '    . $resource->relation) }
		if ($resource->publisher)   { $list .= $cgi->li('Publisher - '   . $resource->publisher) }
		if ($resource->language)    { $list .= $cgi->li('Langauge - '    . $resource->language) }
		if ($resource->type)        { $list .= $cgi->li('Type - '        . $resource->type) }
		if ($resource->format)      { $list .= $cgi->li('Format - '      . $resource->format) }
		if ($resource->date)        { $list .= $cgi->li('Date - '        . $resource->date) }
		if ($resource->coverage)    { $list .= $cgi->li('Coverage - '    . $resource->coverage) }
		if ($resource->contributor) { $list .= $cgi->li('Contributor - ' . $resource->contributor) }
		if ($resource->creator)     { $list .= $cgi->li('Creator - '     . $resource->creator) }
		if ($location)              { $list .= $cgi->li("Location - $location") }
		$list .= $cgi->li("Terms - $related_terms");
		$list  = $cgi->ul($list);
		$html .= $list;
		
	}
    
}



sub delete {

	# initialize the output
	$html = $cgi->h1('Delete resources');
	
	# branch according to the input
	if (! $cgi->param('button')) {
		
		# display a form
		$html .= $cgi->p("Enter the ID of the resource you want to delete. Don't know the ID number? Use <a href='" . $cgi->script_name . "?cmd=find'>find</a>.");
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->textfield(-name => 'id');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'delete', -label => 'Delete');		 
		$html .= $cgi->endform;
		
	}
	
	elsif ($cgi->param('button') eq 'Delete') {
    
    	# create an object based on input
    	my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
    	
		# display a confirmation
		$html .= $cgi->p('Are you sure you want to delete the resource named <b>' . $resource->name . '</b>?');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->hidden(-name => 'id', -value => $resource->id);
		$html .= $cgi->submit(-name => 'button', -value => 'yes', -label => 'Yes');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'no', -label => 'No');		 
		$html .= $cgi->endform;
    
    }
    
    elsif ($cgi->param('button') eq 'Yes') {
    
    	# delete it; do the work and echo the result
    	my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
    	$resource->delete;
    	$html .= $cgi->p('The resource ' . $resource->name . ' has been deleted.');
    
    }
    
    elsif ($cgi->param('button') eq 'No') {
    
    	# abort
    	my $resource = MyLibrary::Resource->new(id => $cgi->param('id'));
    	$html .= $cgi->p('The resource ' . $resource->name . ' has <b>not</b> been deleted.');

    }
    
}


sub reindex {

	# rebuild the index
	my $output;
	open (OUTPUT, REBUILD . " | ") or carp "Can't rebuild: $!";
	while (<OUTPUT>) { $output .= $_ }
	close (OUTPUT);
	
	# display the result, hopefully
	$html .= '<h1>Rebuild</h1>';
	$html .= '<p>The index has been rebuilt.';
	$html .= "The output of the process should be displayed below:</p><pre style='font-size: small'>$output</pre>";

}

sub oai {

	# clean out the existing oai file ; ugly!
	my $cmd = 'rm ' . HTDOCS . OAIROOT . '*.xml';
	system $cmd;
		
	# get all the resources
	my @resource_ids = MyLibrary::Resource->get_resources(output => 'id');
	
	# process each resource
	foreach (@resource_ids) {
	
		# extract what i need
		my $resource = MyLibrary::Resource->new(id => $_);
		my $title = &escape($resource->name);
		my $description = &escape($resource->note);
		my $url; foreach ($resource->resource_locations) { $url = &escape($_->location) }
		
		my @subject_ids = $resource->related_terms();
		my $subjects;
		foreach (@subject_ids) {
		
			my $subject = MyLibrary::Term->new(id => $_);
			$subjects .= '<subject>' . &escape($subject->term_name) . '</subject>';
		
		}
		
		# open, write, and close the file
		open (OAI, '> ' . HTDOCS . OAIROOT . 'mylibrary_portal-' . $resource->id . '.xml');
		print OAI <<EOR;
<?xml version="1.0" encoding="UTF-8"?>

<oaidc:dc xmlns="http://purl.org/dc/elements/1.1/" 
	xmlns:oaidc="http://www.openarchives.org/OAI/2.0/oai_dc/" 
	xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
	xsi:schemaLocation="http://www.openarchives.org/OAI/2.0/oai_dc/
	http://www.openarchives.org/OAI/2.0/oai_dc.xsd">
				  
	<title>$title</title>
	<description>$description</description>
	$subjects
	<relation>This record is a part of the MyLibrary Portal (http://dewey.library.nd.edu/mylibrary/morgan/zagreb/).</relation>

	<identifier>$url</identifier>

</oaidc:dc>
EOR

		close OAI;
	
	}
	
	# display the result, hopefully
	$html .= '<h1>Make OAI-PMH</h1>';
	$html .= "<p>The <a href='" . OAIROOT . "'>OAI files</a> have been (re)created.</p>";
	
}


sub gracefulExit {

	# output the 'magic line', a standard header, the content, and a standard footer, then done
	print $cgi->header;
	print &header;
	print $html;
	print &footer;
	exit;

}


sub escape {

    my $text = shift;
    $text =~ s/&/&amp;/g;
    $text =~ s/</&lt;/g;
    $text =~ s/>/&gt;/g;
    return($text);
    
}


