#!/usr/bin/perl

# manage-facets.pl - a sample MyLibrary script to create, find, edit, and delete facets

# Eric Lease Morgan <emorgan@nd.edu>

# 2006-01-11 - wrote pod
# 2003-10-31 - first cut; Holloween

=head1 NAME

manage-facets.pl - create, find, edit, and delete facets from a MyLibrary database

=head1 DESCRIPTION

This script is one of the simplest in the system because it only deals with one type of data, facets. Moreover, there are only four things you can do here: create a facet, find (and then dump) all facets, edit a facet, and delete a facet. In order to edit or delete facets you will need to know the facet's key (id). That is one of the primary uses of the find all feature.

If you are just starting out learning the MyLibrary system, then this script is a good place to start reading. It is a good introduction to the way the various MyLibrary objects are implemented throughout the module.

=head1 AUTHOR

Eric Lease Morgan

=cut

# include the necessary modules
use lib '../lib/';
use MyLibrary::Facet;
use strict;
require 'subroutines.pl';

# display an introduction
&clearScreen;
print "\nUse this program to create and maintain MyLibrary facets.\n\n";

# initialize menu choice
my $choice eq '';

do {

	# display menu
	print "Facets main menu\n";
	print "================\n";
	print <<EOM;
[c]reate a facet
[f]ind all facets
[e]dit a facet
[d]elete a facet
  
[q]uit
  
EOM
	print "Which one? "; chop ($choice = <STDIN>);
	
	# branch according to chioice; create a facet
	if ($choice eq 'c') {
	
		# get the input
		print "What is the name of this new facet? "; chop (my $name = <STDIN>);
		print "Please describe this facet. "; chop (my $note = <STDIN>);
		
		# create the object, fill it, and save it
		my $facet = MyLibrary::Facet->new;
		$facet->facet_name($name);
		$facet->facet_note($note);
		$facet->commit;
		
		# echo the result
		&listOneFacet ($facet->facet_id);
		
	}
	
	# find all facets
	elsif ($choice eq 'f' ) { &listAllFacets }
	
	# edit a facet
	elsif ($choice eq 'e' ) {
	
		# get the facet to edit
		print "What is the ID of the facet you want to edit? "; chop (my $id = <STDIN>);
		
		# get it
		my $facet = MyLibrary::Facet->new (id => $id);
		
		# check for success
		if ($facet ne undef) {
		
			# get new input
			print "Change the name of the facet from " . $facet->facet_name . " to: "; chop (my $name = <STDIN>);
			print "Change the note of the facet from " . $facet->facet_note . " to: "; chop (my $note = <STDIN>);
			
			# update the object
			$facet->facet_name($name);
			$facet->facet_note($note);
			$facet->commit;

			# echo the result
			&listOneFacet ($facet->facet_id);
			
		}
		
		# error
		else { print "\nERROR: The id, $id, does not point to a valid facet. Try find.\n\n" }
	
	}
	
	# delete a facet
	elsif ($choice eq 'd' ) {
	
		# get the facet to delete
		print "What is the ID of the facet you want to delete? "; chop (my $id = <STDIN>);
		
		# get it
		my $facet = MyLibrary::Facet->new (id => $id);
		
		# check for success
		if ($facet ne undef) {
		
			# display 
			print "\nFacet:\n";
			print "    id = " . $facet->facet_id . "\n";
			print "  name = " . $facet->facet_name . "\n";
			print "  note = " . $facet->facet_note . "\n";
			print "\n";

			# confirm
			print "Do you really want to delete this facet [y/n]? "; chop (my $reply = <STDIN>);
			
			# check reply
			if ($reply eq 'y') {
			
				# delete it
				$facet->delete;
				print "Facet deleted.\n\n";
				
			}
			
		}
		
		# error
		else { print "\nERROR:The id, $id, does not point to a valid facet. Try find.\n\n" }
	
	}
	
	# quit
	elsif ($choice eq 'q' ) { print "\nBye, bye\n" }
	
	else { print "Unknown choice: $choice. Returning to menu.\n" }
	
} until $choice eq 'q';


